<?php

/**
 * CSV_Reporter Class
 * Genera reportes en formato csv, los cuales cumplen con el estandar RFC 4180.
 */

class CSV_Report extends rs{

	/**
	 * Almacena la consulta que va a ejecutarse para obtener los datos...
	**/
	public $query = '';

	/**
	 * Establece el delimitador para los campos (evita problemas cuando los campos
	 * contienen comas, puntos y comas e incluso saltos de línea)
	**/
	public $delimiter = '"';

	/**
	 * Establece el separador para los campos.
	**/
	public $separator = ';';

	/**
	 * Establece los títulos (rótulos de columnas) que se utilizarán en la primer
	 * fila del archivo.
	 * Se le debe asignar un array asociativo donde los keys sean los nombres de los
	 * campos en la Base de datos y los valores sean los rótulos que se desee mostrar.
	 * Nota:
	 * -> El orden de salida de los campos no se basará en el orden que traigan
	 * los campos en el resultado de la consulta, sino que será idéntico al establecido
	 * en este array asociativo.
	 * -> Para que un campo tenga el alias que trae por defecto desde la base de datos
	 * o el establecido en la consulta basta con asignar el valor "-" al campo.
	 * -> Si se desea omitir un campo basta con que no figure en este array.
	 * @example $handler->$field_names = array('cliente_id'=>'Identificador'); cambia
	 * el campo "cliente_id" por el nombre "Identificador".
	 * @example $handler->$filed_names = array('cliente_id'=>'-'); deja el nombre del
	 * campo por defecto (el nombre que viene en el resultado de la consulta).
	**/
	public $field_names = array ();

	/**
	 * Establece los formatos para los distintos campos en el archivo.
	 * Análogo al array anterior pero aquí se establecerá la función de formateo para
	 * cada uno de los campos. Escencialmente sirve para aplicar funciones ya establecidas
	 * (transformar a fecha, transformar booleanos, transformar mediante tablas relacionadas)
	 * o también permite la aplicación de funciones personalizadas definidas en el momento
	 * (opción de formateo tipo "custom").
	 * Ejemplos:
	 * 1. $handler->$field_formats = array(
	 * &nbsp;&nbsp;&nbsp;'cliente_activo' => array('function_type' => 'native', 'function_name' => 'format_as_boolean')
	 * );
	 * ->Esta función se encarga de formatear campos booleanos, utilizando la regla siguiente:
	 * Si no se establecen parametros adicionales (adosando un tercer parámetro "params" como
	 * un array asociativo de índices "true" y "false" con valores para cada uno) se formateará
	 * el campo con "Sí" en caso de existir un "1" o con "No" en caso de existir un "0"...
	 * ->Por el contrario, si se establecen parámetros en el tercer índice "params" de la forma
	 * 'params'=>array('true'=>'Activo', 'false'=>'Inactivo') se formateará el campo añadiendo
	 * "Activo" cuando aparezca un "1" e "Inactivo" cuando aparezca un "0". De esta manera podemos
	 * formatear los valores devueltos a nuestro gusto, lo que brinda una flexibilidad extrema.
	 * 2. $handler->$field_formats = array(
	 * &nbsp;&nbsp;&nbsp;'proyecto_fecha' => array('function_type' => 'native', 'function_name' => 'format_as_date')
	 * );
	 * ->Esta función se encarga de formatear campos de fecha, utilizando la regla siguiente:
	 * Si no se establecen parámetros adicionales (adosando un tercer parámetros "params" como
	 * un string en el que se establece el formato de fechas de php) se formateará la fecha de
	 * manera automática utilizando el fomato "d/m/Y".
	 * Por el contrario, si se establece un formato, ese será el formato que se aplicará.
	 * 3. $handler->$filed-formats = array(
	 * &nbsp;&nbsp;&nbsp;'pais_id' => array('function_type' => 'native', 'function_name' => 'format_with_link', 'params' => array('tabla_relacionada'=>'paises', 'clave_foranea'=>'pais_id', 'campo_solicitado'=>'pais_nombre')
	 * );
	 * ->Esta función se encarga de formatear campos que son claves foraneas (FK) apuntando a tablas
	 * relacionadas, trabaja utilizando los parámetros "params" QUE SON OBLIGATORIOS.
	 * Tales parámetros consisten en un array asociativo que tiene tres índices:
	 * &nbsp;&nbsp;*'tabla_relacionada': Indica el nombre de la tabla a la que apunta la FK.
	 * &nbsp;&nbsp;*'clave_foranea': Indica el nombre del campo al que se apunta en la tabla relacionada.
	 * &nbsp;&nbsp;*'campo_solicitado': Indica el nombre del campo que se desea mostrar u obtener a cambio del actual.
	 * 4. $handler->$filed_formats = array(
	 * &nbsp;&nbsp;&nbsp;'proyecto_url' => array('function_type' => 'custom', 'function_name' => 'custom_function', 'params' => mixed)
	 * );
	 * ->Esta función se encarga de formatear campos utilizando funciones "custom", o sea, personalizadas
	 * las cuales deben incluirse en el archivo que utilice esta clase, se establece el nombre de la función
	 * mediante el índice "function_name" y se pasan los parámetros, que pueden ser del tipo que se quiera o nulos.
	 * Debe tenerse especial cuidado de que los parámetros pasados sean los que la función espera recibir, de lo
	 * contrario es imposible que no se generen errores al utilizar esta característica.
	**/
	public $field_formats = array ();
	/**
	 * Establece si el archivo será descargable automáticamente o si solo se guardará.
	**/
	public $downloadable = true;
	/**
	 * Establece la ruta en que se guardará el archivo en caso de que sea descargable.
	**/
	public $file_path = "media/backups/csv_reports";
	/**
	 * Establece el nombre que se le dará al archivo en caso de que sea descargable.
 	 **/
	public $file_name = "csv_report";

	public $object = array();

	public $header;
	/**
	 * Función que genera un el reporte en formato CSV utilizando la configuración
	 * establecida mediante las variables de clase (nótese el uso de las mismas de
	 * manera pública, es decir, no existen métodos que permitan el seteo de las
	 * mismas, por lo tanto, es de suma importancia que se proporcionen DATOS CORRECTOS
	 * antes de invocar a esta función, o bien se dejen los valores establecidos por
	 * defecto, los cuales darán un archivo funcional pero sin personalizar.
	 */
	public function generate_csv_report(){

		$csv = $this->get_header();

		$csv.= $this->parse_line($this->field_names);

		if($this->query == ""){
			$this->query = $this->last_query;
		}

		$registros = db::query($this->query);

		while($row = db::fetch_assoc($registros)){
			$fields = $this->order_array($row, $this->field_names);
			$csv .= $this->parse_line($this->format_fields($fields));
		}

		if($this->downloadable){

		   header('Content-Description: File Transfer');
		   header("Content-Type: application/csv; charset=UTF-8");
		   header('Content-Type: application/force-download');
		   header('Content-Disposition: attachment; filename='.$this->file_name.".csv");
		   header('Content-Transfer-Encoding: binary');
		   header('Expires: 0');
		   header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
		   header('Pragma: public');
		   header('Content-Length: ' . strlen($csv));
		   echo $csv;
		   exit();

		}else{

			if(file_put_contents($this->file_path."/".$this->file_name.".csv",$csv)){

				return true;

			}else{

				return false;

			}

		}

	}

	public function get_header(){

		if($this->header){
			$fields = count($this->field_names);
			$header[] = $this->header;
			$result = array_pad($header, $fields-1, ";");
			return implode($this->separator, $result)."\n";
		}
	}

	/**
	 * Función que recibe un array con los datos correspondientes a un registro y los
	 * retorna de acuerdo al formato establecido para cada uno de ellos.
	 * @param Array Asociativo $fields Los campos a formatear.
	 * @return Array Numérico El arreglo en que se almacenan los campos formateados
	 */
	private function format_fields($fields){

		$formated_fields = Array(); // Array para ir guardando los campos luego de formatearlos.

		foreach($fields as $key => $value){ // Recorremos cada uno de los campos...

			if(array_key_exists($key, $this->field_names)){

				if(isset($this->field_formats[$key])){ // Si hay un formato definido para el campo actual...

					$control_array = $this->field_formats[$key]; // Variable para almacenar los datos de control.

					$control_type = $control_array["function_type"];
					$function_name = $control_array["function_name"];
					$parameters = "";

					if(isset($control_array["params"])){
						$parameters = $control_array["params"];
					}

					// Se debió cambiar la línea que sigue por las tres líneas precedentes por problemas con la función "list" y los arrays asociativos...
					//list($control_type, $function_name, $parameters) = each($control_array); // Se almacenan los datos del array de control en las variables "ControlType", "FunctionName" y "Parameters"...

					if($control_type === "native"){ // Si el formateo se debe hacer con funciones propias de la presente clase...
						$formated_fields[] = $this->$function_name($value, $parameters);
					}

				}else{ // De lo contrario, no se especificó un formato especial, por lo tanto se retorna el valor tal y como está.
					$formated_fields[] = $value;
				}
			}
		}// end for

		return $formated_fields;
	}

	/**
	 * Función que recibe dos arrays, uno es el array a ordenar, y otro es el array de referencia.
	 * Ordena los campos del array pasado basándose en el orden del array de referencia.
	 * @param Array $array_to_order El array que va a ser ordenado.
	 * @param Array $like_this_array El array de referencia
	 * @return Array El array ordenado.
	**/
	private function order_array($array_to_order, $like_this_array){

		$ordered_array = Array();

		foreach($like_this_array as $key => $values){

			$ordered_array[$key] = $array_to_order[$key];

		}

		return $ordered_array;

	}

	/**
	 * Función que recibe un array de datos y lo transforma en una línea pronta para
	 * adosar al archivo CSV, utilizando los separadores, delimitadores y saltos de lína
	 * establecidos o los seteados por defecto.
	 * @param Array Numérico $line La línea de datos a transformar.
	 * @return String La línea previamente formateada según el estandar CSV.
	**/
	private function parse_line($line){

		foreach($line as $key => $value){

			if($value != ""){

				if($value != "-"){

					$value = str_replace($this->delimiter, $this->delimiter.$this->delimiter, $value);

					$line[$key] = $this->delimiter.$value.$this->delimiter;

				}else{

					$line[$key] = $key;

				}

			}else{

				$line[$key] = "";

			}

		}

		$line = implode($this->separator, $line);

		return $line."\n";

	}

	/**
	 * Format As Boolean
	 *
	 * Función que formatea un valor alojado en la fila de datos cambiando su valor original
	 * (un "0" o un "1") por valores de verdadero y falso que también son pasados como parámetros.
	 * @param Array $params El array que contiene los parámetros para la función en el siguiente orden:</br>
	 * 0 => Boolean El dato que se debe parsear.</br>
	 * 1 => String El valor que se debe retornar en caso de que el valor sea verdadero.</br>
	 * 2 => String El valor que se debe retornar en caso de que el valor sea falso.
	 * @return String El valor correspondiente al valor de verdad del dato recibido.
	**/
	private function format_as_boolean($value, $params){
		if($value == 1){
			if($params != ""){
				return $params["true"];
			}else{
				return "Si";
			}
		}else{
			if($params != ""){
				return $params["false"];
			}else{
				return "No";
			}
		}
	}

	/**
	 * Format As Fecha
	 *
	 * @param String $value El valor de la fecha que se desea formatear.
	 * @return String La fecha formateada a dd/mm/yyyy.
	**/
	private function format_as_date($date, $params){
		if($date === null){
			return false;
		}
		if($params == ""){
			return date_format(date_create($date), "d/m/Y");
		}else{
			return date_format(date_create($date), $params);
		}
	}

	/**
	 * Format With Link
	 *
	 * Función que retorna un valor alojado en una tabla externa relacionado con un campo determinado
	 * en la "tabla principal".
	 * AÑANADIDO por el pelado - Si es necesario mas datos a traer, ej: algun campo combinado de otra tabla o de la misma -
	 * @param Array $params El array que contiene los parámetros para la función en el siguiente orden:</br>
	 * 0 => String El nombre de la tabla con la que se debe hacer el "link" o "join".
	 * 1 => String El nombre del campo que es clave foránea (FK) en dicha tabla.
	 * 2 => Integer El valor que debe tener dicha FK.
	 * 3 => El campo de la tabla que debe ser devuelto al realizar el "linkeo".
	 * @return String El dato correspondiente luego de encontrar el valor en la tabla externa.
	 * Este enjendro se agrando dado los limites existentes en ella, el array $params ahora recibiendo otros paramatros podesmos hacer JOINS o llamar a mas de un campos
	 *
	 *
	 * * @$params['tabla_relacionada']
	 * * @$params['campo_solicitado']
	 * * @$params['clave_foranea']
	 *
	 *  Si es necesario mas datos a traer, ej: algun campo combinado de otra tabla o de la misma
	 *
	 * * @$params['campo_solicitado_extra']
	 * * @$params['tabla_union']
	 * * @$params['campo_union']
	 * * @params['format_date'] - Posi tenes un campo fecha en la solicitud, solamante dejamos este campo asi, toma valor bool
	 * *
	 *
	 *
	 *
	**/
	private function format_with_link($id, $params){
		$return = false;

		if(!is_array($params['campo_solicitado'])){
			$params['campo_solicitado'] = array($params['campo_solicitado']);
		}

		foreach ($params['campo_solicitado'] AS $campo){
				$campo_solicitado[] = db::ftquote($campo);
		}

		//array_walk('db::ftquote', $campo_solicitado);
		$campo_solicitado = implode(",", $campo_solicitado);

		if($params['campo_solicitado_extra']){
			$query.= " SELECT ".$campo_solicitado." FROM ".db::ftquote($params['tabla_relacionada'])." ";
			$query.= " INNER JOIN ".db::ftquote($params['tabla_union'])." ON ".db::ftquote($params['tabla_union']).".".db::ftquote($params['tabla_relacionada'])." = ";
			$query.= db::ftquote($params['tabla_relacionada']).".".db::ftquote($params['campo_solicitado_extra']);
			$query.= " WHERE ".db::ftquote($params['tabla_relacionada']).".".db::ftquote($params['campo_union'])." =  ".db::quote($id)." ";
		}else{
			$query = "SELECT ".$campo_solicitado." FROM ".db::ftquote($params['tabla_relacionada'])." WHERE ".db::ftquote($params['clave_foranea'])."=".db::quote($id);
		}

		if($res = db::query($query)){
			while ($row = db::fetch_array($res)){
				foreach ($params['campo_solicitado'] AS $campo){
					if($params['format_date']){
						$return[] = $this->format_as_date($row[$campo]);
					}else{
						$return[] = $row[$campo];
					}
				}
			}

			if($params['separador']){
				$return = implode($params['separador'], $return);
			}else{
				$return = implode(" ", $return);
			}
		}

		return $return;
	}

	private function format_link_array($id,$params){
		return $params[$id];
	}

	/**
	 *
	 * Pasamos metodos al objeto
	 * @param $id
	 * @param $params['object'],$params['function']
	 */
	private function format_as_class($id,$params){
		$object = $params['object'];
		$other  = $params['params'];
		if(is_object($object)){
			$function = $params['function'];
			if(method_exists($object,"$function")){
				return $object->$function($id,$other);
			}else{
				return false;
			}
		}
	}

	private function parse_fields($field,$params){
		$function = $params['function'];
		$param    = $params['parameters'];

		if(function_exists($function)){
			    $field = preg_replace("[\n|\r|\n\r]",'', $field);
			    $field = html_entity_decode($field,ENT_NOQUOTES,'ISO-8859-15');
			    //$field = htmlspecialchars_decode($field);
			    $field = $function($field,$param);
				return $field;
			}else{
				return $field;
		}
	}

	private function format_as_number($value, $params){
		return number_format($value,''.$params[0].'',''.$params[1].'',''.$params[2].'');
	}

}

?>